using System;
using System.Collections.Generic;
using System.Text;
using EzeScan.Interfaces;

namespace Complete
{
    class Upload_Class : IUpload
    {
        private UploadTargetField[] _uploadFields;
        private UserConfigurableParameter[] _uploadParameters;
        private String[] _fieldTypes;

        public Upload_Class()
        {
            // This sample Plugin supports the following types (The names can be anything - EzeScan just lists them for the user to select)
            _fieldTypes = new String[] {"String","Integer","Boolean"};
            
            // define the field type arrays to limit the field types certain fields can be set to
            String[] string_integer = new String[] { _fieldTypes[0], _fieldTypes[1] };
            String[] boolean_type = new String[] { _fieldTypes[2] };            
            
            //Create the ID field - It's default static value is "1" - It defaults to using a static value - It can only be set to String or Integer type - Default is string
            // - is not user or system mandatory - has no user or system max length - and uses a custom dialog to simplify setting static values            
            UploadTargetField new_field = Connection_Class.PluginHelper.CreateField("ID", "1", true, ref string_integer, string_integer[0],false, false, 0, 0, true);
            Connection_Class.PluginHelper.AddField(ref _uploadFields, ref new_field);
            
            //Create the Increment field - It's default static value is "" - It does not default to using a static value - It can only be set to Boolean type - Default Type is Boolean
            // - is not user or system mandatory - has no user or system max length - and has no custom dialog for setting static values
            new_field = Connection_Class.PluginHelper.CreateField("Increment", "", false, ref boolean_type, boolean_type[0], false, false, 0, 0, false);
            Connection_Class.PluginHelper.AddField(ref _uploadFields, ref new_field);
            
            // Create the parameter that sets the numeric range - it uses a static field custom dialog to set the value
            UserConfigurableParameter new_param = Connection_Class.PluginHelper.CreateParameter("range", "Range Limits", ParameterType.CustomLocked, "1,10", "Enter the limits to use when browsing for an ID range", false);
            Connection_Class.PluginHelper.AddParameter(ref _uploadParameters, ref new_param);
        }

        #region IUpload Members

        bool IUpload.AllowArbitraryFieldNameEntry(ref UserConfigurableParameter[] upload_parameters)
        {
            // EzeScan calls this function to determine whether the user should be allowed to add custom fields
            // Custom fields are usually either extra EDRMS metadata fields created by the use
            // In this sample Plugin custom fields are not handled and therefore not allowed
            return false;
        }

        int IUpload.EditCustomField(string field_name, ref string field_value)
        {
            // EzeScan calls this function when the user has clicked the browse button on a custom field (which has it's source set to static)
    
            Int32 return_code = 0; // Default to code for user cancelled            
            
            switch (field_name)
            {
                case "ID":
                    // Note this is separate from the range limit parameter
                    SelectRange_Dialog select_range_dialog = new SelectRange_Dialog();
                    Int32 selected_range = select_range_dialog.SelectFromRange(1, 10);
                    
                    if (select_range_dialog.UserResponse == System.Windows.Forms.DialogResult.OK)
                    {
                        field_value = selected_range.ToString();
                        return_code = 1;
                    }
                    break;
            }
            
            return return_code;
        }

        int IUpload.EditCustomParameter(string parameter_id, ref UserConfigurableParameter[] upload_parameters, ref string value_out)
        {
            // See Connection_Class.IConnection_EditCustomParameter() for a description of this function
            // This plugin upload object has one custom parameter - range
            try
            {        
                UserConfigurableParameter range_parameter = new UserConfigurableParameter();    
                Int32 return_code = 0; // the user cancelled editing the value
    
                if (Connection_Class.FindParameter("range", ref upload_parameters, ref range_parameter))
                {
                    String[] range_parts = range_parameter.Value.Split(new char[] {','});
                    Int32 range_start = Int32.Parse(range_parts[0]);
                    Int32 range_end = Int32.Parse(range_parts[1]);
                    ConfigureRange_Dialog config_range_dialog = new ConfigureRange_Dialog();

                    if (config_range_dialog.ConfigureRange(range_start, range_end))
                    {
                        value_out = config_range_dialog.RangeStart.ToString() + "," + config_range_dialog.RangeEnd.ToString();
                        return_code = 1; // User accepted change
                    }
                }
            
                return return_code;
            }
            catch (Exception ex)
            {
                Connection_Class.DebugError(ex);
                Connection_Class.DebugLog.LogUserErrorMessage("Error editing range: " + ex.Message);
                return -99; // Error code
            }
        }

        int IUpload.GetUploadFields(ref UploadTargetField[] upload_fields)
        {
            // EzeScan calls this function when it needs the list of fields the Plugin supports
            // If the list can only be determined whilst the Plugin is connected and a connection is not already established then return the code 2 to ask EzeScan to connect
    
            upload_fields = _uploadFields;
    
            return 1; // Success
        }

        int IUpload.GetUploadFieldTypes(ref string[] upload_types)
        {
            // EzeScan calls this function to get a list of names for the types this plugin supports
            // e.g some systems may have a text type, others may call the same thing a string
            // some systems may have an integer type, others may call it a different name or may not support it at all
    
            // The type to use as the default type should be at the start of the list
            upload_types = _fieldTypes;
    
            return 1; // Success
        }

        bool IUpload.UploadDocument(string document_path, string index_file_path, ref UploadTargetField[] upload_fields, ref UserConfigurableParameter[] upload_parameters, ref IStatusProvider upload_status)
        {
            try
            {            
                UserConfigurableParameter range_parameter = new UserConfigurableParameter();
                Int32 range_start = -1;
                Int32 range_end = -1;                
                String document_id = "";
                
                Boolean return_flag = false; // Default to failed
                Boolean increment_requested = false;

                upload_status.ProgessIncrements = 4;
                upload_status.IncrementStatus("Processing Fields");

                for (Int32 field_index = 0; field_index < Connection_Class.PluginHelper.FieldCount(ref upload_fields); field_index++)
                {
                    switch (upload_fields[field_index].FieldName)
                    {
                        case "ID":
                            document_id = upload_fields[field_index].Value;
                            break;
                        case "Increment":
                            
                            switch (upload_fields[field_index].Value.ToUpper())
                            {
                                case "Y": case "T": case "1": case "YES": case "TRUE":
                                    increment_requested = true;
                                    break;
                            }
                            break;       
                    }                    
                }
                
                // Validate the field values
                upload_status.IncrementStatus("Validating Fields");

                if ((document_id == "") || (!increment_requested))
                {
                    if (document_id == "")
                    {
                        document_id = "1";
                    }                    
                
                    if (Connection_Class.FindParameter("range", ref upload_parameters, ref range_parameter))
                    {                        
                        if ((range_parameter.Value != null) && (range_parameter.Value != ""))
                        {
                            String[] range_parts = range_parameter.Value.Split(new char[] {','});
                            range_start = Int32.Parse(range_parts[0]);
                            range_end = Int32.Parse(range_parts[1]);
                        }
                        
                    }
                    
                    if ((document_id != "") && (range_start != -1 || range_end != -1))
                    {
                        if ((Int32.Parse(document_id) < range_start) || (Int32.Parse(document_id) > range_end))
                        {
                            Connection_Class.DebugLog.LogUserErrorMessage("Validation error - Value '" + document_id + "' is outside of the range '" + range_start.ToString() + "' to '" + range_end.ToString() + "'");
                            document_id = "";
                        }                                
                    }
                            
                    String base_folder = System.IO.Path.Combine(Connection_Class.StorageFolder, Connection_Class.UserName);
                    
                    if (document_id != "")
                    {
                        Int32 starting_id = 1;
     
                        while (increment_requested && System.IO.File.Exists(System.IO.Path.Combine(base_folder, document_id + System.IO.Path.GetExtension(document_path))))
                        {
                            starting_id = Int32.Parse(document_id) + 1;
                            document_id = starting_id.ToString();
                        }
                        
                        if (!System.IO.Directory.Exists(base_folder))
                        {
                            Connection_Class.DebugLog.LogPluginMessage( "Creating Folder: " + base_folder, PluginDebugLevel.InfoNormal);
                            System.IO.Directory.CreateDirectory(base_folder);
                        }
                        
                        String target_file_path = System.IO.Path.Combine(base_folder, document_id + System.IO.Path.GetExtension(document_path));
                        
                        if (System.IO.File.Exists(target_file_path))
                        {
                            Connection_Class.DebugLog.LogPluginMessage("Removing existing file: " + target_file_path, PluginDebugLevel.InfoNormal);
                            System.IO.File.Delete(target_file_path);
                        }
                        
                        Connection_Class.DebugLog.LogPluginMessage("Uploading file: " + target_file_path, PluginDebugLevel.InfoNormal);
                        upload_status.IncrementStatus("Uploading Document");
                        System.IO.File.Copy(document_path, target_file_path, false);
                            
                        Connection_Class.PreviousDocumentReference = document_id;
                        
                        return_flag = true;
                    }
                }       
                else
                {
                    // Validation error - Both ID and Increment are set
                    Connection_Class.DebugLog.LogUserErrorMessage("Validation error - Both ID and Increment are set");
                }
                
                return return_flag;                
            }
            catch (Exception ex)
            {                
                Connection_Class.DebugError(ex);
                Connection_Class.DebugLog.LogUserErrorMessage("Error uploading document: " + ex.Message);
                return false; // Upload failed
            }
        }

        UserConfigurableParameter[] IUpload.UploadParameters
        {
            get { return _uploadParameters; }
        }

        #endregion        
    }
}
